/******************************************************************************
 *    FILENAME : Bsp_Motor.c
 *
 *    PURPOSE  : 电机驱动函数接口
 *
 *       Author: 电笔小新                      Created on: 2025年9月12日
******************************************************************************/

#include "User_Include.h"
#include "Bsp_Motor.h"
//============================================================================
// *  Variables & Function_Defines
//============================================================================


//============================================================================
// * Function    :  Motor_Init
// * Purpose     :  Initialize the resource ports required for motor drive.
// * Parameters  :  Null
// * Return      :  Null
//============================================================================
void Motor_Init(void)
{
    /* ===== 步骤1：GPIO配置 ===== */

    EALLOW;  // 解除寄存器写保护

    // Motor_AIN1 初始化
    GpioCtrlRegs.GPBMUX1.bit.GPIO42=0;    //IO口
    GpioCtrlRegs.GPBDIR.bit.GPIO42=1;     //端口设置输出
    GpioCtrlRegs.GPBPUD.bit.GPIO42=1;     //上拉

    // Motor_AIN2 初始化
    GpioCtrlRegs.GPBMUX1.bit.GPIO33=0;    //IO口
    GpioCtrlRegs.GPBDIR.bit.GPIO33=1;     //端口设置输出
    GpioCtrlRegs.GPBPUD.bit.GPIO33=1;     //上拉

    // Motor_BIN1 初始化
    GpioCtrlRegs.GPAMUX1.bit.GPIO3=0;    //IO口
    GpioCtrlRegs.GPADIR.bit.GPIO3=1;     //端口设置输出
    GpioCtrlRegs.GPAPUD.bit.GPIO3=1;     //上拉

    // Motor_BIN2 初始化
    GpioCtrlRegs.GPAMUX1.bit.GPIO2=0;    //IO口
    GpioCtrlRegs.GPADIR.bit.GPIO2=1;     //端口设置输出
    GpioCtrlRegs.GPAPUD.bit.GPIO2=1;     //上拉

    // 启用 ePWM1 时钟
    SysCtrlRegs.PCLKCR0.bit.TBCLKSYNC = 0;   // 暂停所有 ePWM 时基
    SysCtrlRegs.PCLKCR1.bit.EPWM1ENCLK = 1;  // 开启 ePWM1 时钟

    // 配置 GPIO 引脚为 PWM 功能
    GpioCtrlRegs.GPAMUX1.bit.GPIO0 = 1;   // 配置 GPIO0 为 PWM1A
    GpioCtrlRegs.GPAMUX1.bit.GPIO1 = 1;   // 配置 GPIO1 为 PWM1B

    EDIS;   // 恢复寄存器保护

    /* ===== 步骤2：配置 ePWM1 时基模块 ===== */
    // PWM 频率 = SYSCLKOUT / (TBPRD + 1) / (HSPCLKDIV * CLKDIV)
    EPwm1Regs.TBPRD = 6000;             // 周期值 (60MHz / 6000 = 10KHz PWM)
    EPwm1Regs.TBPHS.half.TBPHS = 0;      // 相位偏移
    EPwm1Regs.TBCTR = 0;                 // 计数器清零

    // 时基时钟配置
    EPwm1Regs.TBCTL.bit.CTRMODE = 0;     // 增减计数模式 (0-UP, 1-DOWN, 2-UPDOWN)
    EPwm1Regs.TBCTL.bit.PHSEN = 0;       // 禁用相位加载
    EPwm1Regs.TBCTL.bit.HSPCLKDIV = 0;   // 高速时钟分频 (0=/1, 1=/2)
    EPwm1Regs.TBCTL.bit.CLKDIV = 0;      // 基准时钟分频 (0=/1, 1=/2...)

    /* ===== 步骤3：配置比较模块 ===== */
    // PWM1A 占空比 = CMPA / TBPRD
    EPwm1Regs.CMPA.half.CMPA = 3000;    // 50% 占空比 (3000/6000)

    // PWM1B 占空比 = CMPB / TBPRD
    EPwm1Regs.CMPB = 1500;              // 25% 占空比 (1500/6000)

    /* ===== 步骤4：配置动作限定模块 (关键步骤) ===== */
    // PWM1A 配置：
    EPwm1Regs.AQCTLA.bit.ZRO = 2;        // CTR=0 时设置高 (2=SET)
    EPwm1Regs.AQCTLA.bit.CAU = 1;        // CTR=CMPA 时清除 (1=CLEAR)

    // PWM1B 配置：
    EPwm1Regs.AQCTLB.bit.ZRO = 2;        // CTR=0 时设置高
    EPwm1Regs.AQCTLB.bit.CBU = 1;        // CTR=CMPB 时清除

    /* ===== 步骤5：禁用不需要的模块 ===== */
    // 死区模块 (DB) - 适用于独立输出
    EPwm1Regs.DBCTL.bit.OUT_MODE = 0;    // 禁用死区生成
    EPwm1Regs.DBCTL.bit.IN_MODE = 0;      // 禁用输入控制

    // 事件触发 (ET) - 可选禁用
//    EPwm1Regs.ETSEL.bit.INTSEL = 0;      // 禁用中断触发
//    EPwm1Regs.ETCLR.bit.INT = 1;          // 清除中断标志

    EPwm1Regs.ETSEL.bit.SOCAEN    = 1;               // Enable SOC on A group
    EPwm1Regs.ETSEL.bit.SOCASEL   = ET_CTR_ZERO;     // Select SOC from counter = ZERO
    EPwm1Regs.ETPS.bit.SOCAPRD    = ET_1ST;          // Generate pulse on 1st event


    // 斩波模块 (PC) - 禁用
    EPwm1Regs.PCCTL.bit.CHPEN = 0;        // 禁用斩波生成


    /* ===== 步骤6：启动 PWM ===== */
    EALLOW;
    SysCtrlRegs.PCLKCR0.bit.TBCLKSYNC = 1;  // 启动所有 ePWM 时基
    EDIS;
}


//============================================================================
// * Function    :  Set_Pwm()
// * Purpose     :  Assign to PWM register.
// * Parameters  :  motor_left->Left wheel PWM；motor_right->Right wheel PWM
// * Return      :  Null
//============================================================================
void Set_Pwm(int motor_left,int motor_right)
{
    if(motor_left>0)
    {
        Motor_AIN1_L;
        Motor_AIN2_H;  //前进
    }
    else
    {
        Motor_AIN1_H;
        Motor_AIN2_L;  //后退
    }
    EPwm1Regs.CMPA.half.CMPA = Cal_abs(motor_left);
    if(motor_right>0)
    {
        Motor_BIN1_H;
        Motor_BIN2_L;  //前进
    }
    else
    {
        Motor_BIN1_L;
        Motor_BIN2_H;  //后退
    }
    EPwm1Regs.CMPB=Cal_abs(motor_right);
}

//============================================================================
// * Function    :  Motor_Disable()
// * Purpose     :  Shut down the motor​ .
// * Parameters  :  Null
// * Return      :  Null
//============================================================================
void Motor_Disable(void)
{
        Motor_AIN1_L;
        Motor_AIN2_L;
        Motor_BIN1_L;
        Motor_BIN2_L;
}
//=============================================================================
// End of file.
//=============================================================================

